<?php
// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in (allow all levels)
if (!isset($_SESSION['user_logged_in'])) {
    header('Location: ../index.php');
    exit;
}

// Include database connection
include '../includes/db.php';

// Get shop info
$shop = $conn->query("SELECT * FROM shop_info WHERE id=1")->fetch_assoc();

// Initialize error variable
$error = null;
$success = null;

// Process form submissions
if (isset($_POST['add_customer'])) {
    $name = trim($_POST['name']);
    $phone = trim($_POST['phone']);
    $address = trim($_POST['address']);
    
    if (empty($name) || empty($phone) || empty($address)) {
        $error = 'Sila isi semua maklumat pelanggan!';
    } else {
        // Check if phone already exists
        $stmt = $conn->prepare("SELECT id FROM customers WHERE phone = ?");
        $stmt->bind_param('s', $phone);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error = 'No telefon sudah wujud dalam sistem!';
            $stmt->close();
        } else {
            $stmt->close();
            $stmt = $conn->prepare("INSERT INTO customers (name, phone, address) VALUES (?, ?, ?)");
            $stmt->bind_param('sss', $name, $phone, $address);
            if ($stmt->execute()) {
                $success = 'Pelanggan berjaya ditambah!';
            } else {
                $error = 'Error: ' . $stmt->error;
            }
            $stmt->close();
        }
    }
}

if (isset($_POST['update_customer'])) {
    $id = (int)$_POST['customer_id'];
    $name = trim($_POST['name']);
    $phone = trim($_POST['phone']);
    $address = trim($_POST['address']);
    
    if (empty($name) || empty($phone) || empty($address)) {
        $error = 'Sila isi semua maklumat pelanggan!';
    } else {
        // Check if phone already exists for other customers
        $stmt = $conn->prepare("SELECT id FROM customers WHERE phone = ? AND id != ?");
        $stmt->bind_param('si', $phone, $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error = 'No telefon sudah wujud dalam sistem!';
            $stmt->close();
        } else {
            $stmt->close();
            $stmt = $conn->prepare("UPDATE customers SET name = ?, phone = ?, address = ? WHERE id = ?");
            $stmt->bind_param('sssi', $name, $phone, $address, $id);
            if ($stmt->execute()) {
                $success = 'Maklumat pelanggan berjaya dikemaskini!';
            } else {
                $error = 'Error: ' . $stmt->error;
            }
            $stmt->close();
        }
    }
}

if (isset($_POST['delete_customer'])) {
    $id = (int)$_POST['customer_id'];
    
    // Check if customer has orders
    $stmt = $conn->prepare("SELECT COUNT(*) as order_count FROM orders WHERE customer_id = ?");
    $stmt->bind_param('i', $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $order_count = $result->fetch_assoc()['order_count'];
    $stmt->close();
    
    if ($order_count > 0) {
        $error = 'Tidak boleh padam pelanggan yang mempunyai tempahan!';
    } else {
        $stmt = $conn->prepare("DELETE FROM customers WHERE id = ?");
        $stmt->bind_param('i', $id);
        if ($stmt->execute()) {
            $success = 'Pelanggan berjaya dipadam!';
        } else {
            $error = 'Error: ' . $stmt->error;
        }
        $stmt->close();
    }
}

// Get customers with order statistics
$customers_query = "
    SELECT c.*, 
           COALESCE(COUNT(o.id), 0) as order_count,
           COALESCE(SUM(o.total_sale), 0) as total_spent
    FROM customers c
    LEFT JOIN orders o ON c.id = o.customer_id
    GROUP BY c.id
    ORDER BY c.name ASC
";

$customers = $conn->query($customers_query);

// Check for SQL errors
if (!$customers) {
    $error = "Database error: " . $conn->error;
    error_log("SQL Error in customers.php: " . $conn->error);
}

// Set current page for sidemenu
$current_page = 'customers';
$base_path = '';

// Include header with sidemenu
include '../includes/header.php';
?>

<style>
.card { border: none; box-shadow: 0 2px 8px rgba(0,0,0,0.1); border-radius: 12px; }
.card-header { background: linear-gradient(45deg, #007bff, #0056b3); color: white; border-radius: 12px 12px 0 0 !important; border: none; }
.btn-modern { border-radius: 8px; padding: 8px 16px; font-weight: 500; transition: all 0.3s ease; }
.btn-modern:hover { transform: translateY(-2px); box-shadow: 0 4px 12px rgba(0,0,0,0.15); }
.table th { background: #343a40 !important; color: white !important; border-top: none; font-weight: 600; }
</style>

                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="bi bi-people"></i> Pengurusan Pelanggan</h1>
                    <button class="btn btn-primary btn-modern" data-bs-toggle="modal" data-bs-target="#addCustomerModal">
                        <i class="bi bi-plus-circle"></i> Tambah Pelanggan Baru
                    </button>
                </div>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?= $error ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?= $success ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-people"></i> Senarai Pelanggan</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Bil</th>
                                        <th>Nama</th>
                                        <th>No Tel</th>
                                        <th>Alamat</th>
                                        <th>Jumlah Order</th>
                                        <th>Jumlah Belanja (RM)</th>
                                        <th>Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    if ($customers && $customers->num_rows > 0): 
                                        $i = 1; 
                                        while($row = $customers->fetch_assoc()): 
                                    ?>
                                    <tr>
                                        <td><?= $i++ ?></td>
                                        <td><?= htmlspecialchars($row['name']) ?></td>
                                        <td><?= htmlspecialchars($row['phone']) ?></td>
                                        <td><?= htmlspecialchars($row['address']) ?></td>
                                        <td>
                                            <span class="badge bg-primary"><?= $row['order_count'] ?></span>
                                        </td>
                                        <td><?= number_format($row['total_spent'], 2) ?></td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#editCustomerModal<?= $row['id'] ?>" title="Edit Pelanggan">
                                                    <i class="bi bi-pencil"></i>
                                                </button>
                                                <button class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#deleteCustomerModal<?= $row['id'] ?>" title="Padam Pelanggan">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    
                                    <!-- Modal Edit Pelanggan -->
                                    <div class="modal fade" id="editCustomerModal<?= $row['id'] ?>" tabindex="-1">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <form method="post">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title">Edit Pelanggan</h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <input type="hidden" name="customer_id" value="<?= $row['id'] ?>">
                                                        <div class="mb-3">
                                                            <label class="form-label">Nama Pelanggan</label>
                                                            <input type="text" name="name" class="form-control" value="<?= htmlspecialchars($row['name']) ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">No Tel</label>
                                                            <input type="text" name="phone" class="form-control" value="<?= htmlspecialchars($row['phone']) ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Alamat</label>
                                                            <textarea name="address" class="form-control" rows="3" required><?= htmlspecialchars(str_replace(["\r\n", "\r", "\n"], "\n", $row['address'])) ?></textarea>
                                                        </div>
                                                    </div>
                                                    <div class="modal-footer">
                                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                                                        <button type="submit" name="update_customer" class="btn btn-primary">Simpan Perubahan</button>
                                                    </div>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Modal Delete Pelanggan -->
                                    <div class="modal fade" id="deleteCustomerModal<?= $row['id'] ?>" tabindex="-1">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <form method="post">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title">Padam Pelanggan</h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <input type="hidden" name="customer_id" value="<?= $row['id'] ?>">
                                                        <p>Adakah anda pasti mahu padam pelanggan <strong><?= htmlspecialchars($row['name']) ?></strong>?</p>
                                                        <p class="text-danger"><small>Tindakan ini tidak boleh dibatalkan.</small></p>
                                                    </div>
                                                    <div class="modal-footer">
                                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                                                        <button type="submit" name="delete_customer" class="btn btn-danger">Padam</button>
                                                    </div>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                    <?php 
                                        endwhile; 
                                    else: 
                                    ?>
                                        <tr>
                                            <td colspan="7" class="text-center text-muted">
                                                <i class="bi bi-inbox"></i> Tiada data pelanggan dijumpai
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Modal Tambah Pelanggan -->
                <div class="modal fade" id="addCustomerModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <form method="post">
                                <div class="modal-header">
                                    <h5 class="modal-title">Tambah Pelanggan Baru</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                </div>
                                <div class="modal-body">
                                    <div class="mb-3">
                                        <label class="form-label">Nama Pelanggan</label>
                                        <input type="text" name="name" class="form-control" required>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">No Tel</label>
                                        <input type="text" name="phone" class="form-control" required>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Alamat</label>
                                        <textarea name="address" class="form-control" rows="3" required></textarea>
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                                    <button type="submit" name="add_customer" class="btn btn-primary">Tambah Pelanggan</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
<?php include '../includes/footer.php'; ?> 